<?php
// manual_csv_import.php - Manual CSV import for UserManager users
require_once 'config.php';

if ($_POST['action'] ?? '' === 'import_csv') {
    $csv_data = $_POST['csv_data'] ?? '';
    
    if (empty($csv_data)) {
        die('No CSV data provided');
    }
    
    $lines = explode("\n", trim($csv_data));
    $db = getDB();
    $imported = 0;
    $errors = 0;
    
    echo "<h3>Importing UserManager Users...</h3>";
    
    // Package mapping
    $package_mapping = [
        11 => 'profile name-11GH',
        15 => 'profile name-15GH',
        22 => 'profile name-22GH', 
        42 => 'profile name-42GH',
        61 => 'profile-61GH'
    ];
    $amounts = array_keys($package_mapping);
    
    foreach ($lines as $line_num => $line) {
        $line = trim($line);
        if (empty($line)) continue;
        
        $data = str_getcsv($line);
        
        // Support different CSV formats:
        // Format 1: username,password
        // Format 2: username,password,amount
        // Format 3: username,password,amount,profile_name
        
        if (count($data) >= 2) {
            $username = trim($data[0]);
            $password = trim($data[1]);
            
            // Determine amount and profile
            if (count($data) >= 3 && is_numeric(trim($data[2]))) {
                $amount = floatval(trim($data[2]));
                $profile = $package_mapping[$amount] ?? 'profile name-11GH';
            } else {
                // Assign random amount
                $amount = $amounts[array_rand($amounts)];
                $profile = $package_mapping[$amount];
            }
            
            if (count($data) >= 4) {
                $profile = trim($data[3]);
            }
            
            $voucher_code = 'MKR-' . strtoupper(substr(md5($username), 0, 8));
            
            echo "<p>Importing: $username / $password (GHS $amount)... ";
            
            try {
                $stmt = $db->prepare("INSERT INTO vouchers (voucher_code, username, password, amount, profile_name, status) VALUES (?, ?, ?, ?, ?, 'available')");
                $stmt->bind_param('sssds', $voucher_code, $username, $password, $amount, $profile);
                
                if ($stmt->execute()) {
                    echo "<span style='color: green;'>✅ SUCCESS</span></p>";
                    $imported++;
                } else {
                    echo "<span style='color: orange;'>⚠️ DUPLICATE</span></p>";
                    $errors++;
                }
                $stmt->close();
            } catch (Exception $e) {
                echo "<span style='color: red;'>❌ ERROR: " . $e->getMessage() . "</span></p>";
                $errors++;
            }
        } else {
            echo "<p style='color: red;'>❌ Line " . ($line_num + 1) . ": Invalid format</p>";
            $errors++;
        }
    }
    
    echo "<hr><h3>Import Results:</h3>";
    echo "<p style='color: green; font-size: 18px;'>✅ Successfully imported: $imported users</p>";
    echo "<p style='color: red; font-size: 18px;'>❌ Errors: $errors</p>";
    
    exit;
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Manual CSV Import</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 0 10px rgba(0,0,0,0.1); max-width: 800px; }
        textarea { width: 100%; height: 300px; margin: 10px 0; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-family: monospace; }
        button { padding: 12px 30px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; }
        button:hover { background: #0056b3; }
        .example { background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 10px 0; border-left: 4px solid #007bff; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Manual CSV Import from UserManager</h1>
        
        <div class="example">
            <h3>📋 CSV Format Instructions:</h3>
            <p><strong>Supported Formats:</strong></p>
            <ul>
                <li><code>username,password</code> (random amount will be assigned)</li>
                <li><code>username,password,amount</code></li>
                <li><code>username,password,amount,profile_name</code></li>
            </ul>
            <p><strong>Example:</strong></p>
            <pre>
john_doe,pass123,11,profile name-11GH
jane_smith,secret456,15,profile name-15GH
mike_brown,hello789,22,profile name-22GH
sarah_white,test000,42,profile name-42GH
admin_user,admin123,61,profile-61GH</pre>
        </div>

        <form method="POST">
            <h3>Paste UserManager Users (CSV format):</h3>
            <textarea name="csv_data" placeholder="username,password,amount,profile_name
john_doe,pass123,11,profile name-11GH
jane_smith,secret456,15,profile name-15GH
...">username,password,amount,profile_name
john_doe,pass123,11,profile name-11GH
jane_smith,secret456,15,profile name-15GH
mike_brown,hello789,22,profile name-22GH
sarah_white,test000,42,profile name-42GH
admin_user,admin123,61,profile-61GH</textarea>
            <br><br>
            <button type="submit" name="action" value="import_csv">🚀 Import Users</button>
        </form>
    </div>
</body>
</html>